// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "main.h"
#include "cpu.h"
#include "container.h"
#include "recompiler.h"
#include "version.h"
#include "ExtendedTrace.h"
#include "stringCommons.h"
#include <pngdib.h>
#include <sys/timeb.h>
#include <process.h>
#include <dxerr9.h>
#include <time.h>
#include <d3dx9.h>

#define PAD_DEFAULTS(macro) \
	macro(a, DIK_A)\
	macro(b, DIK_X)\
	macro(x, DIK_S)\
	macro(y, DIK_Q)\
	macro(z, DIK_Z)\
	macro(start, DIK_RETURN)\
	macro(l, DIK_W)\
	macro(r, DIK_E)\
	macro(aup, DIK_UP)\
	macro(aleft, DIK_LEFT)\
	macro(adown, DIK_DOWN)\
	macro(aright, DIK_RIGHT)\
	macro(dup, DIK_T)\
	macro(dleft, DIK_F)\
	macro(ddown, DIK_G)\
	macro(dright, DIK_H)\
	macro(cup, DIK_I)\
	macro(cleft, DIK_J)\
	macro(cdown, DIK_K)\
	macro(cright, DIK_L)\

namespace g {
#define STANDARD_BOOLS(macro) \
	macro(degub_msg, "dmsg")\
	macro(beep, "beep")\
	macro(degub_run, "fid")\
	macro(dumpmem, "dumpmem")\
	/*macro(dual_run, "dual")*/\
	macro(bouehr, "bouehr")\
	macro(dump_rec_bin_asm, "drba")\
	macro(lowmem_log, "lowmem")\
	macro(advanced_mode, "amode")\
	macro(dump_audio, "daudio")\
	macro(quiet, "quiet")\
	macro(verbose, "verbose")\
	macro(log_interrupts, "lint")\
	macro(cache_enabled, "cache")\
	macro(nerf_sc, "nerf_sc")\
	macro(gcube, "gcube")\
	macro(log_audio, "alog")\
	macro(gp_dtex, "dtex")\
	macro(gp_dtexraw, "dtexraw")\
	macro(software_yuyv, "soft2d")\
	macro(open_edo_on_load, "edo")\
	macro(gp_log, "glog")\
	macro(use_map, "map")\
	macro(gp_wireframe, "gp_wireframe")\
	macro(gp_flat, "gp_flat")\
	macro(gp_white, "gp_white")\
	macro(disassembly_mode, "disasm")\
	macro(exi_log, "elog")\
	macro(memcard_log, "mclog")\
	macro(bba_log, "blog")\
	macro(rec_log, "rlog")\
	macro(gp_frame_dump, "gfdump")\
	macro(si_log, "slog")\
	macro(always_on_top, "aot")\
	macro(dsp_log, "dsplog")\
	macro(fifo_log, "flog")\
	macro(rename_degub, "rename")\
	macro(dvd_log, "dvdlog")\
	macro(wii, "wii")\
	macro(dsp_hle, "dsp_hle")\

	//used in dumpGlobalVariables
#define NONSTANDARD_BOOLS(macro) macro(recompiler, 0)\
	macro(hle, 0)\
	macro(hrr, 0)\
	macro(rec_onestep, 0)\
	macro(rec_disasm, 0)\
	macro(autopause, 0)\
	macro(iloop, 0)\
	macro(edo_osr, 0)\
	macro(use_mdvd, 0)\
	macro(frame_limit, 0)\
	macro(separate_emudebug_file, 0)\

#define STANDARD_MENU_BOOLS(macro) macro(IDM_QUIET, g::quiet)\
	macro(IDM_FRAMELIMIT, g::frame_limit)\
	macro(IDM_FLAT, g::gp_flat)\
	macro(IDM_WHITE, g::gp_white)\

#define DEFINE_STANDARD_BOOL(cname, cmdstring) ,cname=false

	bool recompiler=true, hle=true, hrr=false, rec_onestep=false, gp_pixelshader=true,
		rec_disasm=false, autopause=false, iloop=false, frame_limit=true,
		edo_osr=true, gp_dlrec=true STANDARD_BOOLS(DEFINE_STANDARD_BOOL);
	UINT degub_skip=0, dual_skip=0, audio_buffer_min_size_ms=400;
	UINT ip_src=0, ip_dst_old=0, ip_dst_new=0;
	BYTE mac_address[6] = { 1, 2, 3, 4, 5, 6 };
#ifndef PUBLIC_RELEASE
	std::string mdvd_filename = "F:\\temp\\DolphinSDK1.0\\dvddata.gcm";
	bool use_mdvd=true, degub_on=true, separate_emudebug_file=true;
#else
	std::string mdvd_filename = "";
	bool use_mdvd=false, degub_on=false, separate_emudebug_file=false;
#endif
	TimingModes timing_mode = TM_EXACT_FAST;//TM_REALTIME;
	const char *str_tm[_TM_NMODES] = { TIMING_MODES(STRINGIZE) };
	MEMCARDSLOT_INFO memcardslot_info[2] = {
#ifdef PUBLIC_RELEASE
		{ MSC_MC59, "default.mci", false },
#else
		{ MSC_Empty, "", false },
#endif
		{ MSC_Empty, "", false }
	};

	bool pad_on[PAD_NPADS] = { true, false, false, false };
	PAD_SCANCODES pad_sc[PAD_NPADS];
	BYTE pad_half = DIK_LSHIFT, pad_quarter = DIK_LCONTROL, reset_button = DIK_BACKSPACE;

	DWORD rtcf;
};

CApp g_capp;
IniSystem g_ini("whinecube.ini");
BYTE g_degub_write_mask=0;
#define DWM_VERSION 1
#define DWM_GVARS 2

//These fully control the partitioning of the status bar.
enum StatusBarPart { /*SB_TIMING,*/ SB_CYCLES, SB_MIPS, SB_DSP_MIPS,
SB_FRAMES, SB_FPS, SB_MODE,
SB_NPARTS };
int g_parts[SB_NPARTS] = { 230, 330, 440, 520, 580, -1 };

bool ProcessMessages();
HWND init_window(HINSTANCE hInst);
HWND CreateStatusBar(HWND hwndParent, int nStatusID, HINSTANCE hinst,
										 int nParts, int *parts);
void dumpGlobalVariables();

CRITICAL_SECTION g_uefcs;

void dumpCurrentDate(FILE *file) {
	__time64_t utc_time;
	_time64(&utc_time);
	int daylight;
	_get_daylight(&daylight);
	__time64_t cet_time = utc_time + 3600*(1+daylight);  //+ 1 hour
	tm time;
	_gmtime64_s(&time, &cet_time);
	strftime(g_uefbuf, UEFBUFSIZE, "Current date: %b %d %Y %H:%M:%S CET\n", &time);
	etfprint(file, g_uefbuf);
}

//Has no error handling.
//I think that if an error occurs here there's no way to handle it anyway.
LONG WINAPI MyUnhandledExceptionFilter(LPEXCEPTION_POINTERS e) {
	//EnterCriticalSection(&g_uefcs);

	FILE* file=NULL;
	fopen_s(&file, "exceptioninfo.txt", "a");
	fseek(file, 0, SEEK_END);
	etfprint(file, "\n");
	etfprint(file, g_buildtime);
	etfprint(file, "\n");
	dumpCurrentDate(file);
	etfprintf(file, "Unhandled Exception\n  Code: 0x%08X\n",
		e->ExceptionRecord->ExceptionCode);
	STACKTRACE2(file, e->ContextRecord->Eip, e->ContextRecord->Esp, e->ContextRecord->Ebp);
	fclose(file);
	_flushall();

	//LeaveCriticalSection(&g_uefcs);
	return EXCEPTION_CONTINUE_SEARCH;
}

int WINAPI WinMain(HINSTANCE hInst, HINSTANCE, LPSTR lpszCmdLine, int) {
	try {
		//#ifndef _DEBUG
		EXTENDEDTRACEINITIALIZE(".");
		SetUnhandledExceptionFilter(&MyUnhandledExceptionFilter);
		//int *p = NULL;
		//*p = 1337;
		//#endif
		GLE(g_capp.run(hInst, lpszCmdLine));
		EXTENDEDTRACEUNINITIALIZE();
	} catch(exception &e) {
		handleException(e);
	}
	return 0;
}

void handleException(const exception& e) {
	g_capp.m.wmp_error = true;
	char buffer[MAX_PATH];
	sprintf(buffer, "Way Unhandled Exception: %s\n", e.what());
	THE_ONE_MESSAGE_BOX(buffer);
	g_capp.PostCloseMessage();
}

bool CApp::parseCmdLine(LPSTR lpszCmdLine) {
	char buffer[MAX_PATH]="";
	int i=0, j;

#define PARSE_STANDARD_BOOL(cname, cmdstring) else\
	if(strcmp(buffer, "/" cmdstring) == 0) g::cname = true;

	DEGUB("Command line: %s\n", lpszCmdLine);
	while(sscanf_s(lpszCmdLine + i, " %s%n", buffer, sizeof(buffer), &j) > 0) {
		if(sscanf_s(buffer, "/skip%i", &g::degub_skip) == 1)
			g::degub_run = true;
		//else if(sscanf_s(buffer, "/duals%i", &g::dual_skip) == 1)
		//g::dual_run = true;

		STANDARD_BOOLS(PARSE_STANDARD_BOOL)

		else if(strcmp(buffer, "/rec") == 0)
		g::recompiler = true;
		else if(strcmp(buffer, "/degub") == 0)
			g::degub_on = true;
		else if(strcmp(buffer, "/interp") == 0)
			g::recompiler = false;
		else if(strcmp(buffer, "/nohle") == 0)
			g::hle = false;
		else if(strcmp(buffer, "/nofl") == 0)
			g::frame_limit = false;
		else if(strcmp(buffer, "/nops") == 0)
			g::gp_pixelshader = false;
		else if(strcmp(buffer, "/nodlr") == 0)
			g::gp_dlrec = false;
		else if(strcmp(buffer, "/rt") == 0)
			g::timing_mode = g::TM_REALTIME;
		else if(strcmp(buffer, "/er") == 0)
			g::timing_mode = g::TM_EXACT_REAL;
		else if(strcmp(buffer, "/ef") == 0)
			g::timing_mode = g::TM_EXACT_FAST;
		else if(strcmp(buffer, "/drbad") == 0) {
			g::dump_rec_bin_asm = true;
			g::rec_onestep = true;
		} else if(strcmp(buffer, "/recd") == 0) {
			g::recompiler = true;
			g::rec_disasm = true;
		} else if(strcmp(buffer, "/clog") == 0) {
			g::bba_log = g::exi_log = g::gp_log = g::log_audio = g::memcard_log = g::si_log =
				true;
		} else if(strcmp(buffer, "/gcm") == 0) {
			i += j;
			sscanf_s(lpszCmdLine + i, " %n", &j);
			i += j;
			if(lpszCmdLine[i] == '\"') {
				g::mdvd_filename = "";
				DWORD d=0;
				while(lpszCmdLine[i+1+d] != '\"') {
					g::mdvd_filename += lpszCmdLine[i+1+d];
					d++;
				}
				i += d+2 - j;
			} else {
				sscanf_s(lpszCmdLine + i, " %s%n", buffer, &j);
				g::mdvd_filename = buffer;
			}
			g::use_mdvd = true;
		} else if(buffer[0] == '\"' && !m.started_from_cmd_line) {
			DWORD d=0;
			while(lpszCmdLine[i+1+d] != '\"') {
				m_exename += lpszCmdLine[i+1+d];
				d++;
			}
			i += d+2 - j;
			m.started_from_cmd_line = true;
		} else if(buffer[0] != '/' && !m.started_from_cmd_line) {
			m_exename = buffer;
			m.started_from_cmd_line = true;
		} else {
			string s = "Unknown argument: " + string(buffer) + "\n";
			DEGUB(s.c_str());
			THE_ONE_MESSAGE_BOX(s.c_str());

			if(IsWindow(m.hWnd)) {
				TGLE(DestroyWindow(m.hWnd));
				while(ProcessMessages()) ;
			}
			return false;
		}
		i += j;
	}
	return true;
}

void final_overrides() {
	if(g::dump_rec_bin_asm) {
		g::recompiler = true;
		g::rec_disasm = true;
	}
	if(g::use_map) {
		g::recompiler = false;
	}
	if(g::advanced_mode) {
		g::cache_enabled = false;
	}
	if(g::lowmem_log)
		g::recompiler = false;
	if(g::disassembly_mode) {
		g::degub_run = true;
		g::hle = false;
		g::recompiler = false;
		//g::dual_run = false;
		g::autopause = false;
		if(g::timing_mode == g::TM_REALTIME)
			g::timing_mode = g::TM_EXACT_REAL;
		g::frame_limit = false;
	}
	/*if(g::dual_run) {
	g::rec_onestep = true;
	g::hle = false;
	g::rec_disasm = true;
	g::autopause = false;
	if(g::timing_mode == g::TM_REALTIME)
	g::timing_mode = g::TM_EXACT_REAL;
	g::frame_limit = false;
	}*/
	if(g::degub_run) {
		g::rec_onestep = true;
		g::rec_disasm = true;
		if(g::timing_mode == g::TM_REALTIME)
			g::timing_mode = g::TM_EXACT_REAL;
		g::frame_limit = false;
	}
	if(g::rec_disasm)
		g::rec_log = true;

	dumpGlobalVariables();
}

void dumpGlobalVariables() {
	if(g::degub_on) {
#define DUMP_G_BOOL(cname, cmdstring) { if(g::cname) DEGUB(" %s", #cname); }

		DEGUB("Options:");
		NONSTANDARD_BOOLS(DUMP_G_BOOL);
		STANDARD_BOOLS(DUMP_G_BOOL);
		DEGUB("\n");

		if(g::degub_run)
			DEGUB("degub_skip = %i\n", g::degub_skip);
		//if(g::dual_run)
		//DEGUB("dual_skip = %i\n", g::dual_skip);

		DEGUB("audio_buffer_min_size_ms = %i\n", g::audio_buffer_min_size_ms);
		DEGUB("pad_on: %i %i %i %i\n", g::pad_on[0], g::pad_on[1], g::pad_on[2], g::pad_on[3]);
		DEGUB("timing_mode: %s\n", g::str_tm[g::timing_mode]);

		for(int i=0; i<2; i++) {
			DEGUB("MemCardSlot %i: %s, \"%s\", Caching %s\n", i+1,
				str_MSC_short[g::memcardslot_info[i].content],
				g::memcardslot_info[i].filename.c_str(),
				abled(g::memcardslot_info[i].caching));
		}
		if(g::use_mdvd) {
			DEGUB("DVD image: %s\n", g::mdvd_filename.c_str());
		}
		DEGUB("\n");
		g_degub_write_mask |= DWM_GVARS;
	}
}

extern bool test_muldiv();

bool dumpSystemTimeAdjustment() {
	DEGUB("\nTimer testing:\n");
	DWORD sta, sti;
	BOOL stad;
	TGLE(GetSystemTimeAdjustment(&sta, &sti, &stad));
	DEGUB("SystemTimeAdjustment: %i *100ns\n"
		"SystemTimeIncrement: %i *100ns\n"
		"SystemTimeAdjustmentDisabled: %i\n", sta, sti, stad);
	{
		int i=0;
		DWORD start = GetTickCount(), stop;
		do {
			stop = GetTickCount();
			i++;
		} while(start == stop);
		DEGUB("GetTickCount() test loop ran %i times, %i ms\n", i, stop - start);
	}

	{
		int i=0;
		LARGE_INTEGER start, stop, hrpf;
		TGLE(QueryPerformanceFrequency(&hrpf));
		DEGUB("HighPerformanceFrequency: %I64i Hz. Maximum timing: %I64i seconds (approx. %I64i years).\n",
			hrpf.QuadPart, 0x7FFFFFFFFFFFFFFF / hrpf.QuadPart,
			0x7FFFFFFFFFFFFFFF / (hrpf.QuadPart * 60*60*(24*365 + 6)));  //365.25

		TGLE(QueryPerformanceCounter(&start));
		do {
			TGLE(QueryPerformanceCounter(&stop));
			i++;
		} while(start.QuadPart == stop.QuadPart);
		DEGUB("QueryPerformanceCounter() test loop ran %i times, %I64i ticks\n",
			i, stop.QuadPart - start.QuadPart);
		DEGUB("QueryPerformanceCounter() returned %I64i (0x%016I64X)\n",
			stop.QuadPart, stop.QuadPart);
	}
	{
		Timing timing("Minimum Timing");
	}
	//TGLE(test_muldiv());

	DEGUB("\n");

	return true;
}

void dumpVersion() {
	if(g::degub_on) {
		DEGUB("Version: r%i", RELEASE_NUMBER);
#ifdef BETA_VERSION
		DEGUB("b%s", BETA_VERSION);
#endif
		DEGUB("\n%s\n", g_buildtime);
		dumpCurrentDate(df);
		g_degub_write_mask |= DWM_VERSION;
	}
}

void dumpStuffForRestart() {
	if(!(g_degub_write_mask & DWM_VERSION))
		dumpVersion();
	if(!(g_degub_write_mask & DWM_GVARS))
		dumpGlobalVariables();
	g_degub_write_mask = 0;
}

bool CApp::run(HINSTANCE hInst, LPSTR lpszCmdLine) {
	dumpVersion();

	m.hAccTable = LoadAccelerators(hInst, "IDA_ACCEL1");

	//Get High-Res Counter Frequency
	LARGE_INTEGER hrcf;
	TGLE(QueryPerformanceFrequency(&hrcf));
	if(hrcf.HighPart) {
		DEGUB("QueryPerformanceFrequency() returned %I64i (0x%016I64X)\n",
			hrcf.QuadPart, hrcf.QuadPart);
		FAIL(UE_BAD_HRCF);
	} else {
		g::rtcf = hrcf.LowPart;
	}

	TGLE(dumpSystemTimeAdjustment());

	//Global variables
#define SET_PAD0_DEFAULT(id, value) g::pad_sc[0].id = value;
	PAD_DEFAULTS(SET_PAD0_DEFAULT);
	ZeroMemory(&g::pad_sc[1], sizeof(g::pad_sc) - sizeof(PAD_SCANCODES));

	TGLE(g_ini.DoIniFile());

	//cmdline switches override ini file
	if(!parseCmdLine(lpszCmdLine))
		return true;

	final_overrides();

	if(!g::degub_on) {
		g_degub.turn_off();
		g_degub_write_mask = 0;
	}

	//Window and D3D
	g_hWnd = m.hWnd = init_window(hInst);
	if(g::degub_msg) {
		DEGUB("init_window complete\n");
	}

	if(!initD3D()) {
		DoGLE();
		TGLE(DestroyWindow(m.hWnd));
		return true;
	}
	if(g::degub_msg) {
		DEGUB("initD3D complete\n");
	}

	if(!initDI()) {
		DoGLE();
		TGLE(DestroyWindow(m.hWnd));
		return true;
	}
	if(g::degub_msg) {
		DEGUB("initDI complete\n");
	}

	TGLE(setMenuGreyness());
	TGLE(setMenuChecks());
	if(g::degub_msg) {
		DEGUB("SetMenu* complete\n");
	}
	ProcessMessages();
	if(g::degub_msg) {
		DEGUB("ProcessMessages complete\n");
	}

	//Command-line load
	if(m_exename[0] != 0) {
		TGLE(createCores());
		TGLE(startThread());
	}

	// Da main loop
	if(g::degub_msg) {
		DEGUB("Main loop start\n");
	}
	int res;
	do {
		MSG msg;
		res = GetMessage(&msg, NULL, 0, 0);
		TGLECUSTOM(res, -1);
		if(res == 0)  //WM_QUIT
			break;

		if(g::degub_msg) {
			DEGUB("Message recieved in main loop\n");
		}
		if(!IsDialogMessage(g_capp.m.hwndDegubber, &msg) &&
			!IsDialogMessage(g_capp.m.hwndDebugOutput, &msg))
		{
			if(!TranslateAccelerator(m.hWnd, m.hAccTable, &msg)) {
				DispatchMessage(&msg);
			}
		}
	} while(res);
	if(g::degub_msg) {
		DEGUB("Main loop stop\n");
	}
	return true;
}

bool CApp::cCcore() {
	const char *gcmname = g::mdvd_filename.c_str();

	CORE_ARGUMENTS a = { m.hWnd, m.pd3dDevice, m.pdiDevice };
	/*if(g::dual_run) {
	m.cpu = new CPU(a);
	m.cpu1 = new Recompiler(a);

	if(g::use_mdvd) {
	TGLE(m.cpu->getHardware().loadgcm(gcmname));
	m.cpu->getHardware().setlid(false);
	TGLE(m.cpu1->getHardware().loadgcm(gcmname));
	m.cpu1->getHardware().setlid(false);
	}

	TGLE(m.cpu->load(m_exename.c_str()));
	TGLE(m.cpu1->load(m_exename.c_str()));
	} else*/ {
		if(g::recompiler)
			m.cpu = new Recompiler(a);
		else
			m.cpu = new CPU(a);

		if(g::use_mdvd) {
			TGLE(m.cpu->getHardware().loadgcm(gcmname));
			m.cpu->getHardware().setlid(false);
		}

		TGLE(m.cpu->load(m_exename.c_str()));
	}
	return true;
}

bool CApp::createCores() {
	MYASSERT(m.cpu == NULL);
	Timing timing("createCores()");
	SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_CYCLES, (LPARAM)"Loading...");

	m.temp_no_autopause = false;

	char base[MAX_PATH], ext[MAX_PATH], windowtext[0x400], degubname[0x400];
	_splitpath_s(m_exename.c_str(), NULL, 0, NULL, 0, base, sizeof(base),
		ext, sizeof(ext));
	if(g::rename_degub) {
		sprintf(degubname, "degub.%s%s.txt", base, ext);
		TGLE(g_degub.restart(degubname));
		dumpStuffForRestart();
	}

	if(!cCcore()) {
		DWORD error = GetLastError();
		SAFE_DELETE(m.cpu); SAFE_DELETE(m.cpu1);
		SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_CYCLES, (LPARAM)"");
		FAIL(error);
	}

	m.hw = &m.cpu->getHardware();
	m.frames = m.old_frames = 0;
	m.old_cycles = 0;

	if(_stricmp(ext, ".gcm") == 0) {
		sprintf(degubname, "degub.%s.%s.txt", m.cpu->getGameName(), m.cpu->getGameCode());
		sprintf(windowtext, APP_NAME " - %s (%s)", m.cpu->getGameName(),
			m.cpu->getGameCode());
	} else {
		sprintf(degubname, "degub.%s%s.txt", base, ext);
		sprintf(windowtext, APP_NAME " - %s%s", base, ext);
	}
	if(g::rename_degub) {
		TGLE(g_degub.rename(degubname));
	}
	SetWindowText(m.hWnd, windowtext);

	if(g::open_edo_on_load) {
		TGLE(createDebugOutput());
	}
	if(m.hwndDebugOutput) {
		TGLE(PostMessage(m.hwndDebugOutput, LM_LOAD, 0, 0));
	}

	TGLE(setMenuChecks());
	TGLE(setMenuGreyness());
	return true;
}

bool CApp::deleteCores() {
	if(m.cpu) {
		DEGUB("vsync count: %i vct, %i hct, %i flip, %i interrupt read, %i EFB copy\n",
			m.vst_count[VST_VCT_READ], m.vst_count[VST_HCT_READ], m.vst_count[VST_FBA_WRITE],
			m.vst_count[VST_INTERRUPT_READ], m.vst_count[VST_EFB_COPY]);
		ZeroMemory(m.vst_count, sizeof(m.vst_count));
	}
	m.hw = NULL;
	SAFE_DELETE(m.cpu); SAFE_DELETE(m.cpu1);

	for(int i=0; i<SB_NPARTS; i++) {
		TGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, i, (LPARAM)""));
	}
	if(m.hwndDebugOutput) {
		TGLE(PostMessage(m.hwndDebugOutput, LM_LOAD, 0, 0));
	}
	TGLE(SetWindowText(m.hWnd, APP_NAME));
	TGLE(setMenuGreyness());

	m.emu_error = false;
	return true;
}

enum EmuThreadResult { ETR_SUCCESS, ETR_EMU_ERROR, ETR_EXCEPTION,
_ETR_STILL_ACTIVE=STILL_ACTIVE };

unsigned __stdcall EmuThreadFunc(void *) {
	g_capp.m.thread_running = true;
	EmuThreadResult result = ETR_SUCCESS;
	DEGUB("Thread begun\n");
	try {
		if(!g_capp.runemu())
			result = ETR_EMU_ERROR;
	} catch(exception &e) {
		g_errorstring = "Caught exception in thread: "+ string(e.what()) +"\n";
		GLE(PostMessage(g_capp.getHWnd(), LM_ERROR_STRING, 0, 0));
		//BFE(s);
		result = ETR_EXCEPTION;
	}
	GLE(PostMessage(g_capp.getHWnd(), LM_THREAD_FINISH, 0, 0));
	DEGUB("Thread ended\n");
	g_capp.m.thread_running = false;
	return result;
}

bool CApp::startThread() {
	MYASSERT(!m.running);
	ProcessMessages();  //may delete the Cores
	if(!m.cpu)
		return true;
	TGLE(m.hThread = (HANDLE)_beginthreadex(NULL, 0, EmuThreadFunc, NULL, 0, NULL));
	m.running = true;
	TGLE(SetThreadPriority(m.hThread, THREAD_PRIORITY_BELOW_NORMAL));
	TGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"0"));
	//DEGUB("SwitchToThread() returned %i\n", SwitchToThread());
	return true;
}

bool CApp::quitThread() {
	if(!m.running) return true;

	DWORD res;
	{
		Timing timing("quitThread() core");
		static int number = 0;
		number++;
		//DEGUB("quitThread %i\n", number);

		m.cpu->setQuitSignal("quitThread");
		TGLE(MyResumeThread(m.hThread));
		res = WaitForSingleObject(m.hThread, 5000);  //arbitrary timeout
		TGLECUSTOM(WAIT_FAILED, res);
		m.running = false;
		if(res == WAIT_TIMEOUT) {
			DEGUB("%i\t", number);
			//TGLE(TerminateThread(m.hThread, 0));
			FAIL(UE_THREAD_TIMEOUT);
		}
		TGLE(GetExitCodeThread(m.hThread, &res));
		if(res == STILL_ACTIVE) {
			FAIL(STILL_ACTIVE);
		}
		if(g::verbose) {
			MYFILETIME creation, exit, kernel, user;
			TGLE(GetThreadTimes(m.hThread, &creation.ft, &exit.ft, &kernel.ft, &user.ft));
			DEGUB("Emu Thread:\n");
			dumpFileTime("Lifetime", exit.qword - creation.qword);
			dumpFileTime("Kernel time", kernel.qword);
			dumpFileTime("User time", user.qword);
			DEGUB("\n");
		}
		if(g::verbose) {
			MYFILETIME creation, exit, kernel, user;
			TGLE(GetThreadTimes(GetCurrentThread(), &creation.ft, &exit.ft, &kernel.ft,
				&user.ft));
			DEGUB("Main Thread:\n");
			dumpFileTime("Lifetime", exit.qword - creation.qword);
			dumpFileTime("Kernel time", kernel.qword);
			dumpFileTime("User time", user.qword);
			DEGUB("\n");
		}
		TGLE(CloseHandle(m.hThread));
	}
	//dumptimingsifverbose();
	/*if(g::verbose) {
	std::vector<TIMING> timings;
	m.cpu->getTimings(timings);
	//ct.core.QuadPart -= m.ui_timing.QuadPart;
	__int64 sum = 0;
	for(size_t i=0; i<timings.size(); i++) {
	//MYASSERT((sum.QuadPart + cur[i]->QuadPart) > sum.QuadPart
	sum += timings[i].t;
	}
	if(sum <= 0) {
	DEGUB("ERROR:\n");
	} else {
	DEGUB("Total Timings:\n");
	}
	for(size_t i=0; i<timings.size(); i++) {
	if(sum <= 0) {
	DEGUB("%s: %I64i ticks\n", timings[i].str, timings[i].t);
	} else {
	__int64 PERCENTAGE = ((timings[i].t * 100) / sum);
	DEGUB("%s: %I64i%% | %I64i ticks\n", timings[i].str, PERCENTAGE, timings[i].t);
	}
	}
	}*/
	//m.ui_timing.QuadPart = m.old_ui_timing.QuadPart = 0;
	if(res != ETR_SUCCESS) {
		m.emu_error = true;
		HR(updateFrameBuffer());
		TGLE(InvalidateRect(m.hWnd, NULL, false));
		//if(!m.snapshot_taken) {
		//TGLE(takeSnapshot(true));
		//}
	} /*else if(m.started_from_cmd_line) {
		PostCloseMessage();
		}*/
	TGLE(setMenuGreyness());
	TGLE(setMenuChecks());
	TGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"Paused"));

	return true;
}

long CApp::_LM_THREAD_FINISH(HWND, WPARAM, LPARAM) {
	if(m.hw && !m.hw->gp_activated())
		updateScreenAndKillTimer();
	R0GLE(quitThread());
	return 0;
}
long CApp::_LM_ERROR_STRING(HWND, WPARAM, LPARAM) {
	m.temp_no_autopause = true;
	THE_ONE_MESSAGE_BOX(g_errorstring);
	if(m.started_from_cmd_line) {
		PostCloseMessage();
	}
	return 0;
}
long CApp::_LM_ERROR_GLE(HWND, WPARAM, LPARAM lParam) {
	m.temp_no_autopause = true;
	SET_AND_SHOW_LE(lParam);
	if(m.started_from_cmd_line) {
		PostCloseMessage();
	}
	return 0;
}
long CApp::_LM_GPEXCEPTION(HWND, WPARAM, LPARAM) {
	m.temp_no_autopause = true;
	R0GLE(quitThread());
	return 0;
}

bool CApp::runemu() {
	bool error=false;//, error1=false;
	//DEGUB("runemu1: %i\n", m.cpu->quit());

	GLE(setMenuGreyness());
	GLE(setMenuChecks());

	/*if(g::dual_run) {
	Timing timing("Dual run");
	while(!m.cpu->quit()) {
	//degub_on = false;
	//m.cpu->setstop(false);
	error = !m.cpu->run(1, g::degub_run, g::degub_skip,
	false);
	//if(_status87() & (_SW_UNDERFLOW | _SW_OVERFLOW | _SW_ZERODIVIDE | _SW_INVALID)) {
	//DEGUB("FPU error in Interp! Status baaz: 0x%X\n", _status87());
	//error = true;
	//} else {
	error1 = !m.cpu1->run(1, g::degub_run, g::degub_skip,
	false);
	//if(_status87() & (_SW_UNDERFLOW | _SW_OVERFLOW | _SW_ZERODIVIDE | _SW_INVALID)) {
	//DEGUB("FPU error in Rec! Status Register: 0x%X\n", _status87());
	//error = true;
	//}
	//}
	if(error || error1) {
	if(error) {
	DEGUB("An Interp error occurred: %s\n", g_errorstring);
	}
	if(error1) {
	DEGUB("A Rec error occurred: %s\n", g_errorstring);
	}
	DEGUB("%I64i cycles run\n", m.cpu->getCycles());
	error = error || error1;
	break;
	}

	if(g::dual_skip < m.cpu->getCycles()) {
	if(m.cpu->getRegisters().compare_and_degub(m.cpu1->getRegisters()) ||
	m.cpu->getNIA() != m.cpu1->getNIA() ||
	m.cpu->getCycles() != m.cpu1->getCycles())
	{
	DEGUB("interp->nia=%08X\nrec->nia=%08X\n", m.cpu->getNIA(),
	m.cpu1->getNIA());
	DEGUB("after cycle %I64i/%I64i\n", m.cpu->getCycles(),
	m.cpu1->getCycles());
	DEGUB("Old registers:\n");
	m_old_dual_regs.degub_all();
	DEGUB("Interpreter registers:\n");
	m.cpu->getRegisters().degub_all();
	DEGUB("Recompiler registers:\n");
	m.cpu1->getRegisters().degub_all();

	break;
	}
	m_old_dual_regs.copy(m.cpu->getRegisters());
	}
	}
	DEGUB("%I64i cycles run\n", m.cpu->getCycles());
	} else*/ if(g::disassembly_mode) {
		error = m.cpu->disassemble();
	} else
		error = !m.cpu->run(MAX_DWORD, g::degub_run, g::degub_skip, g::dumpmem);
	//DEGUB("runemu2: %i\n", m.cpu->quit());

	if(error) {
		if(GetLastError() == NO_ERROR) {
			PostMessage(m.hWnd, LM_ERROR_STRING, 0, 0);
		} else {
			PostMessage(m.hWnd, LM_ERROR_GLE, 0, GetLastError());
		}
	}

	//if(IsWindow(m.hWnd)) {
	GLE(setMenuGreyness());
	GLE(setMenuChecks());
	//}

	if(g::disassembly_mode)
		return false;

	return !error;
}

#define SWITCH_MESSAGE(mess) case mess: return g_capp._##mess(hWnd, wParam, lParam);

long CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	try {
		//LARGE_INTEGER throwaway;
		//AdditiveTiming(&(g_capp.m.running ? g_capp.m.ui_timing : throwaway));
		if(g::degub_msg) {
			DEGUB("Message:%s. wParam=%lX. lParam=%lX.\n",
				TranslateWM(message).c_str(), wParam, lParam);
		}

		switch(message) {
			WM_MESSAGES(SWITCH_MESSAGE)
		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
		}
	} catch(exception &e) {
		g_capp.m.wmp_error = true;
		char buffer[MAX_PATH];
		sprintf(buffer, "Way Unhandled Exception: %s\n", e.what());
		THE_ONE_MESSAGE_BOX(buffer);
		g_capp.PostCloseMessage();
	}
	return 0;
}

void CApp::updateScreenAndKillTimer() {
	//if(m.bActive) {
	HR(updateFrameBuffer());
	GLE(InvalidateRect(m.hWnd, NULL, false));
	//}
	KillTimer(m.hWnd, SCREEN_TIMER);
}

long CApp::_WM_CREATE(HWND hWnd, WPARAM, LPARAM) {
	R0GLE(m.hwndStatusBar = CreateStatusBar(hWnd, 0, GetModuleHandle(NULL), SB_NPARTS,
		g_parts));
	if(g::verbose) {
		RECT rect;
		R0GLE(GetWindowRect(hWnd, &rect));
		//put it on the right side of the main window, below the EDO window
		R0GLE(m.hwndVDO = CreateWindow("EDIT", "", ES_MULTILINE | WS_VISIBLE | WS_BORDER,
			rect.left + WIDTH + GetSystemMetrics(SM_CXBORDER)*6, rect.top + 32 + 400,
			320, 240, hWnd, NULL, GetModuleHandle(NULL), NULL));
		R0GLE(m.hfontCourierNew = CreateFont(-12, 0, 0, 0, 0, false, false, false,
			ANSI_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
			FIXED_PITCH | FF_MODERN, "Courier New"));
		SendMessage(m.hwndVDO, WM_SETFONT, (WPARAM)m.hfontCourierNew, FALSE);
		//R0GLE(ShowWindow(m.hwndVDO, SW_SHOWNORMAL));
		//give focus back to the main window
		R0GLE(SetActiveWindow(hWnd));
	}
	DragAcceptFiles(hWnd, true);
	R0GLE(SetTimer(hWnd, CYCLE_TIMER, CYCLE_INTERVAL, NULL));

	return 0;
}
long CApp::_WM_TIMER(HWND, WPARAM wParam, LPARAM) {
	switch(wParam) {
	case SCREEN_TIMER:
		{
			static int frames_old;
			frames_old = m.frames;
		}
		if(m.running && !m.lsync_since_last_timer) {
			//m.hw->vi_simple_interrupt();
			//if(m.bActive) {
			R0HR(updateFrameBuffer());
			R0GLE(InvalidateRect(m.hWnd, NULL, false));
			//}
		}
		m.lsync_since_last_timer = false;
		break;
	case CYCLE_TIMER:
		if(m.cpu && m.running) {
			//The reset button
			if(m.bActive) {
				DIDEVICEOBJECTDATA didod[10];
				DWORD              dwElements = 10;
				R0HR(m.pdiDevice->GetDeviceData(sizeof(DIDEVICEOBJECTDATA), didod,
					&dwElements, NULL));
				for(DWORD i=0; i<dwElements; i++) {
					if((didod[i].dwData & 0x80) && didod[i].dwOfs == g::reset_button) {
						m.hw->reset_button_pressed();
					}
				}
			}
			//The status bar
			string s = make_space3(asString(m.cpu->getCycles())) + " cycles";
			R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_CYCLES, (LPARAM)s.c_str()));
			if(m.recompiling) {
				R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS,
					(LPARAM)"Recompiling..."));
			} else if(m.cpu && !m.running) {
				R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"Paused"));
			} else {
				R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)
					ips_string(m.cpu->getCycles() - m.old_cycles, CYCLE_INTERVAL).c_str()));
			}
			R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_DSP_MIPS, (LPARAM)
				("DSP: "+ips_string(m.hw->getDSPCycles() -
				m.old_dsp_cycles, CYCLE_INTERVAL)).c_str()));
			s = asString(m.frames) + " frames";
			R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_FRAMES, (LPARAM)s.c_str()));
			s = asString(m.frames - m.old_frames) + " FPS";
			R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_FPS, (LPARAM)s.c_str()));
			//R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_TIMING,
			//(LPARAM)"Timings disabled"));
#if 0
			if(g::dual_run) {
				R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_TIMING,
					(LPARAM)"Dual Mode; timings disabled"));
			} else {
				std::vector<TIMING> cur;
				m.cpu->getTimings(cur);
				ct.core.QuadPart -= m.ui_timing.QuadPart;
				const size_t NTIMINGS = 6;
				const char *str[NTIMINGS] = { "Core", "HW", "VI", "GX", "DSP", "UI" };
				const LARGE_INTEGER *cur[NTIMINGS] = { &ct.core, &ct.hw.main,
					&ct.hw.vi, &ct.hw.gx, &ct.hw.dsp, &m.ui_timing };
				LARGE_INTEGER *old[NTIMINGS] = { &m.old_ct.core, &m.old_ct.hw.main,
					&m.old_ct.hw.vi, &m.old_ct.hw.gx, &m.old_ct.hw.dsp, &m.old_ui_timing };*//*
					__int64 sum = 0;
				std::vector<__int64> diff;
				if(m_old_timings.size() != cur.size())
					m_old_timings.resize(cur.size(), 0);
				for(size_t i=0; i<cur.size(); i++) {
					diff.push_back(cur[i].t - m_old_timings[i]);
					sum += diff[i];
					m_old_timings[i] = cur[i].t;
				}
				VDEGUB("Timings: %s | %i FPS\n",
					ips_string(m.cpu->getCycles() - m.old_cycles, CYCLE_INTERVAL).c_str(),
					m.frames - m.old_frames);
				if(sum == 0) {
					strcpy(buffer, "Zero");
					//VDEGUB("%s\n", buffer);
				} else {
					buffer[0] = 0;
					for(size_t i=0; i<cur.size(); i++) {
#define PERCENTAGE ((diff[i] * 100) / sum)
						VDEGUB("%s: %I64i%% | %I64i ticks | %I64i ticks total\n",
							cur[i].str, PERCENTAGE, diff[i], cur[i].t);
						sprintf(buffer, "%s%s%I64i %s", buffer,
							(i==0)?"":", ", PERCENTAGE, cur[i].str);
#undef PERCENTAGE
					}
					R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_TIMING, (LPARAM)buffer));
				}
			}
#endif	//0
			R0GLE(InvalidateRect(m.hwndStatusBar, NULL, FALSE));
			if(g::verbose) {
				ostringstream str;
				m.cpu->getVerboseText(str);
				m.hw->getVerboseText(str);
				s = str.str();
				string s2;
				s2.reserve(s.size());
				for(size_t i=0; i<s.size(); i++) {
					if(s[i] == '\n')
						s2 += '\r';
					s2 += s[i];
				}
				//DEGUB("%s", buffer);
				//fails for no reason with ERROR_INVALID_WINDOW_HANDLE
				/*R0GLE*/(SendMessage(m.hwndVDO, WM_SETTEXT, 0, (LPARAM)s2.c_str()));
			}

			m.old_cycles = m.cpu->getCycles();
			m.old_dsp_cycles = m.hw->getDSPCycles();
			m.old_frames = m.frames;
		}
		break;
	}
	return 0;
}
long CApp::_LM_RECOMPILATION_START(HWND, WPARAM, LPARAM) {
	if(g::beep) {
		R0GLE(Beep(150, 100));
	}
	if(!m.recompiling) {
		R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"Recompiling..."));
		if(m.hw && !m.hw->gp_activated())
			updateScreenAndKillTimer();
		m.recompiling = true;
		R0GLE(setMenuGreyness());
	}
	return 0;
}
long CApp::_LM_RECOMPILATION_FINISH(HWND hWnd, WPARAM, LPARAM) {
	if(g::beep) {
		R0GLE(Beep(150, 100));
	}
	R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"0"));
	if(m.hw) if(!m.hw->gp_activated())
		R0GLE(SetTimer(hWnd, SCREEN_TIMER, INIT_INTERVAL, NULL));
	m.recompiling = false;
	R0GLE(setMenuGreyness());
	return 0;
}
long CApp::_LM_VSYNC(HWND hWnd, WPARAM wParam, LPARAM) {
	if(wParam == VST_EFB_COPY && g::gp_frame_dump)
		takeSnapshot(false);

	//Timing timing("LM_VSYNC");
	m.vst_count[wParam]++;
	m.frames++;
	if(g::beep) {
		R0GLE(Beep(100, 100));
	}
	//if(m.bActive) {
	if(m.hw && !m.hw->gp_activated()) {	//2D/xfb grapics
		R0HR(updateFrameBuffer());
		R0GLE(InvalidateRect(hWnd, NULL, false));
	} else {  //3D/gp/gx graphics
		//... We don't need to do anything here, do we?
	}
	//}
	m.lsync_since_last_timer = true;
	return 0;
}
long CApp::_LM_GP_ACTIVATED(HWND, WPARAM, LPARAM) {
	//updateScreenAndKillTimer();	//causes crash, I think
	KillTimer(m.hWnd, SCREEN_TIMER);
	char buffer[32];
	sprintf(buffer, "3D %s", m.hw->getGfxModeString());
	R0GLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MODE, LPARAM(buffer)));
	return 0;
}

long CApp::_WM_ACTIVATE(HWND, WPARAM wParam, LPARAM) {
	// Set m.bActive
	if(LOWORD(wParam) == WA_INACTIVE) {
		if(m.running) {
			if(g::autopause && !m.recompiling && !m.temp_no_autopause) {
				DEGUB("Autopausing\n");
				R0GLE(quitThread());
				m.autopaused = true;
				DEGUB("Autopaused\n");
			}
			if(!g::autopause) {
				SetPriorityClass(GetCurrentProcess(), BELOW_NORMAL_PRIORITY_CLASS);
			}
			m.hw->setWindowActive(false);
		}
		m.bActive = false;
	} else if(!HIWORD(wParam)) {
		m.bActive = true;
		if(m.pdiDevice) {
			R0HR(m.pdiDevice->Acquire());
		}
		if(m.cpu) {
			R0HR(updateFrameBuffer());
			if(m.autopaused && !m.running) {
				DEGUB("Autoresuming\n");
				R0GLE(startThread());
				m.autopaused = false;
				DEGUB("Autoresumed\n");
			}
			if(!g::autopause) {
				SetPriorityClass(GetCurrentProcess(), NORMAL_PRIORITY_CLASS);
			}
			if(m.hw)
				m.hw->setWindowActive(true);
		}
	}

	return 0;
}

#define WMP_ERROR_HANDLER _WM_CLOSE(hWnd, 0, 0); return 0;
#define WMPGLE(func) { if((func) == (0)) { m.wmp_error = true;\
	DO_GLE; WMP_ERROR_HANDLER; } }
#define WMPHR(func) { HRESULT _hr; if(FAILED(_hr=(func))) { m.wmp_error = true;\
	IN_FILE_ON_LINE; DoDXErr(_hr); WMP_ERROR_HANDLER; } }

long CApp::_WM_PAINT(HWND hWnd, WPARAM wParam, LPARAM lParam) {
	if(m.wmp_error)
		return DefWindowProc(hWnd, WM_PAINT, wParam, lParam);;
	//Timing timing("WM_PAINT Outer");
	bool do_paint = (GetUpdateRect(hWnd, NULL, false) != 0);
	//if(GetLastError() != NO_ERROR) {
	//R0GLE(0);
	//}
	if(do_paint) {
		PAINTSTRUCT ps;
		bool clear = true;
		WMPGLE(BeginPaint(hWnd, &ps));

		if(m.hw) if(m.hw->gp_activated()) {  //3D
			if(m.emu_error) {
				WMPHR(m.hw->gp_endScene());
				WMPHR(m.hw->gp_clear());
				WMPHR(m.pd3dDevice->Present(NULL, NULL, NULL, NULL));
			}
			clear = false;
		} else /*if(m.hw->gfx_initialized())*/ { //2D
			SuspendThread(m.hThread);	//kinda hacky
			char buffer[32];
			sprintf(buffer, "2D %s", m.hw->getGfxModeString());
			WMPGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MODE, LPARAM(buffer)));
			//Timing timing("WM_PAINT Inner");
			LPDIRECT3DSURFACE9 pd3dsBackbuffer;
			WMPHR(m.pd3dDevice->GetBackBuffer(0, 0, D3DBACKBUFFER_TYPE_MONO,
				&pd3dsBackbuffer));
			WMPHR(m.pd3dDevice->StretchRect(m.pd3dSurface, NULL, pd3dsBackbuffer, NULL,
				D3DTEXF_NONE));
			pd3dsBackbuffer->Release();
			RECT r = { 0, 0, WIDTH, HEIGHT };
			WMPHR(m.pd3dDevice->Present(NULL, &r, NULL, NULL));
			/*{
			static int counter = 0;
			if(counter > 10) {
			//throwGLE(UE_GENERIC_ERROR);
			wmp_error = true;
			SET_AND_SHOW_LE(UE_GENERIC_ERROR);
			GLE(quitThread()); GLE(deleteCores());
			} else
			counter++;
			}*/
			clear = false;

			//Reset the timer to minimize flicker
			WMPGLE(SetTimer(hWnd, SCREEN_TIMER, INIT_INTERVAL, NULL));
			ResumeThread(m.hThread);	//kinda hacky
		}
		if(clear && m.pd3dDevice) {
			WMPHR(m.pd3dDevice->Clear(0, NULL, D3DCLEAR_TARGET, 0, 0, 0));
			WMPHR(m.pd3dDevice->Present(NULL, NULL, NULL, NULL));
			WMPGLE(InvalidateRect(m.hwndStatusBar, NULL, true));
		}

		WMPGLE(EndPaint(hWnd, &ps));
	}

	return 0;
}

HRESULT CApp::updateFrameBuffer() {
	if(m.hw) if(!m.hw->gp_activated()) {
		if(g::software_yuyv) {
			Container<BYTE> con(640*480*2);
			D3DLOCKED_RECT locked_rect;
			THR(m.pd3dSurface->LockRect(&locked_rect, NULL, D3DLOCK_DISCARD));
			if(m.running) { //an attempt to work around the freeze bug
				SuspendThread(m.hThread);
			}
			m.hw->getScreen(con);
			if(m.running) {
				ResumeThread(m.hThread);
			}
			convertYUYV2XRGB((BYTE *)locked_rect.pBits, con);
			THR(m.pd3dSurface->UnlockRect());
		} else {
			//Timing timing("UpdateFrameBuffer()");
			D3DLOCKED_RECT locked_rect;
			if(m.running) { //an attempt to work around the freeze bug
				SuspendThread(m.hThread);
				//if(SuspendThread(m.hThread) == -1)
				//if(GetLastError() != ERROR_INVALID_HANDLE)
				//GLE(0);
			}
			THR(m.pd3dSurface->LockRect(&locked_rect, NULL, D3DLOCK_DISCARD));
			m.hw->getScreen((BYTE *)locked_rect.pBits);
			THR(m.pd3dSurface->UnlockRect());
			if(m.running) {
				ResumeThread(m.hThread);
				//if(ResumeThread(m.hThread) == -1)
				//if(GetLastError() != ERROR_INVALID_HANDLE)
				//GLE(0);
			}
		}
	}
	return S_OK;
}

bool CApp::initD3D() {
	if(NULL == (m.pD3D = Direct3DCreate9(D3D_SDK_VERSION))) {
		FAIL(UE_DIRECT3D);
	}

	{
		D3DADAPTER_IDENTIFIER9 d3dai;
		HR2GLE(UE_D3DDEVICE, m.pD3D->GetAdapterIdentifier(D3DADAPTER_DEFAULT, 0, &d3dai));
		DEGUB("D3D Adapter Identifier:\n");

#define D3DAI_STRING(id) DEGUB("%s: %s\n", #id, d3dai.id)
		D3DAI_STRING(Driver);
		D3DAI_STRING(Description);
		D3DAI_STRING(DeviceName);

		DEGUB("Version: %u.%u.%u.%u\n", HIWORD(d3dai.DriverVersion.HighPart),
			LOWORD(d3dai.DriverVersion.HighPart), HIWORD(d3dai.DriverVersion.LowPart),
			LOWORD(d3dai.DriverVersion.LowPart));

#define D3DAI_UINT(id) DEGUB("%s: %u (0x%X)\n", #id, d3dai.id, d3dai.id)
		D3DAI_UINT(VendorId);
		D3DAI_UINT(DeviceId);
		D3DAI_UINT(SubSysId);
		D3DAI_UINT(Revision);

		DEGUB("GUID: %08X.%04X.%04X", d3dai.DeviceIdentifier.Data1,
			d3dai.DeviceIdentifier.Data2, d3dai.DeviceIdentifier.Data3);
		for(int i=0; i<8; i++) {
			DEGUB(".%02X", d3dai.DeviceIdentifier.Data4[i]);
		}
		DEGUB(" (");
		for(int i=0; i<8; i++) {
			DEGUB("%c", d3dai.DeviceIdentifier.Data4[i]);
		}
		DEGUB(")\n");

		DEGUB("WHQLLevel: ");
		if(d3dai.WHQLLevel == 0) {
			DEGUB("Not certified.");
		} else if(d3dai.WHQLLevel == 1) {
			DEGUB("Certified, but no date information is available.");
		} else {
			DEGUB("Certified. Year: %u  Month: %u  Day: %u", HIWORD(d3dai.WHQLLevel),
				HIBYTE(LOWORD(d3dai.WHQLLevel)), LOBYTE(LOWORD(d3dai.WHQLLevel)));
		}

		DEGUB("\n\n");
	}

	D3DPRESENT_PARAMETERS d3dpp;
	ZeroMemory(&d3dpp, sizeof(d3dpp));
	d3dpp.Windowed = TRUE;
	d3dpp.BackBufferFormat = D3DFMT_X8R8G8B8;//MY_BACKBUFFERFORMAT;//D3DFMT_UNKNOWN;
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
	d3dpp.PresentationInterval = D3DPRESENT_INTERVAL_IMMEDIATE;
	d3dpp.BackBufferWidth = WIDTH;
	d3dpp.BackBufferHeight = HEIGHT;
	d3dpp.EnableAutoDepthStencil = TRUE;
	d3dpp.AutoDepthStencilFormat = D3DFMT_D24X8;
	d3dpp.Flags = D3DPRESENTFLAG_LOCKABLE_BACKBUFFER;

	D3DDISPLAYMODE d3ddm;
	HR2GLE(UE_D3DDEVICE, m.pD3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT, &d3ddm));

	m.d3dFormat = d3ddm.Format;
	//d3dpp.BackBufferFormat = m.d3dFormat;

	/*HR2GLE(UE_INSUFFICIENT_D3D_CAPS, m.pD3D->CheckDeviceType(D3DADAPTER_DEFAULT,
	D3DDEVTYPE_HAL, m.d3dFormat, MY_BACKBUFFERFORMAT, TRUE));
	HR2GLE(UE_INSUFFICIENT_D3D_CAPS, m.pD3D->CheckDeviceFormat(D3DADAPTER_DEFAULT,
	D3DDEVTYPE_HAL, m.d3dFormat, D3DUSAGE_RENDERTARGET, D3DRTYPE_SURFACE,
	d3dpp.BackBufferFormat));
	HR2GLE(UE_INSUFFICIENT_D3D_CAPS, m.pD3D->CheckDeviceFormat(D3DADAPTER_DEFAULT,
	D3DDEVTYPE_HAL, m.d3dFormat, D3DUSAGE_DEPTHSTENCIL, D3DRTYPE_SURFACE,
	d3dpp.AutoDepthStencilFormat));
	HR2GLE(UE_INSUFFICIENT_D3D_CAPS, m.pD3D->CheckDepthStencilMatch(D3DADAPTER_DEFAULT,
	D3DDEVTYPE_HAL, m.d3dFormat, d3dpp.BackBufferFormat, d3dpp.AutoDepthStencilFormat));*/
	//give warning instead of error!

	/*D3DCAPS9 caps;
	R0HR(m.pD3D->GetDeviceCaps(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, &caps));
	if(caps.MaxActiveLights < 8) {
	DUMPINT(caps.MaxActiveLights);
	R0GLE(UE_INSUFFICIENT_D3D_CAPS);
	}*/

	//do lots of caps checking here, to make sure that all wyrd things we do will work.
	//select software or hardware vertex processing based on the results.

	/*HRESULT hr = m.pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, m.hWnd,
	D3DCREATE_MIXED_VERTEXPROCESSING | D3DCREATE_MULTITHREADED, &d3dpp, &m.pd3dDevice);
	//D3DCREATE_HARDWARE_VERTEXPROCESSING | //D3DCREATE_FPU_PRESERVE |
	if(FAILED(hr)) {
	DEGUB("CreateHardwareDevice Error: %s\nTrying to create a software device... ",
	DXGetErrorString9(hr));*/

	/*HR2GLE(UE_D3DDEVICE, m.pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL,
	m.hWnd, D3DCREATE_SOFTWARE_VERTEXPROCESSING | //D3DCREATE_FPU_PRESERVE |
	D3DCREATE_MULTITHREADED, &d3dpp, &m.pd3dDevice));*/
	HRESULT hr = m.pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL,
		m.hWnd, D3DCREATE_SOFTWARE_VERTEXPROCESSING | //D3DCREATE_FPU_PRESERVE |
		D3DCREATE_MULTITHREADED, &d3dpp, &m.pd3dDevice);
	if(FAILED(hr)) {
		INFOL_DX(hr);
		//set up sub-standard device. 3d ops will most likely fail,
		//but it should be enough to run any 2d programs on an old nVidia Riva TNT2.
		d3dpp.BackBufferFormat = D3DFMT_UNKNOWN;
		d3dpp.EnableAutoDepthStencil = FALSE;
		HR2GLE(UE_D3DDEVICE, m.pD3D->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL,
			m.hWnd, D3DCREATE_SOFTWARE_VERTEXPROCESSING | //D3DCREATE_FPU_PRESERVE |
			D3DCREATE_MULTITHREADED, &d3dpp, &m.pd3dDevice));
	}
	/*DEGUB("Succeded.\n");
	if(m.pd3dDevice == NULL)
	R0GLE(UE_GENERIC_ERROR);
	}*/
	const char *profile = D3DXGetPixelShaderProfile(m.pd3dDevice);
	DEGUB("PixelShaderProfile: %s\n", profile);
	if(!profile) {
		THE_ONE_MESSAGE_BOX("Your graphics card does not support pixel shaders. "
			"WhineCube will run without them, but many commercial games will not work. "
			"Expect errors containing \"Unemulated TEV\".");
		g::gp_pixelshader = false;
		TGLE(g_ini.WriteIniFile());
	}
	return createSurface();
}

bool do_gdi_thing(HWND) {
	g::software_yuyv = true;
	DEGUB("Software yuyv conversion activated!\n");
	THE_ONE_MESSAGE_BOX("Your graphics card doesn't support hardware YUYV conversion. "
		"Software will be used instead.");
	TGLE(g_ini.WriteIniFile());
	return true;
}

bool CApp::createSurface() {
	if(!g::software_yuyv) {
		HRESULT hr;
		hr = m.pD3D->CheckDeviceFormat(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL,
			m.d3dFormat, 0, D3DRTYPE_SURFACE, D3DFMT_YUY2);
		if(hr == D3DERR_NOTAVAILABLE) {
			//SET_AND_SHOW_LE(UE_INSUFFICIENT_D3D_CAPS);
			TGLE(do_gdi_thing(m.hWnd));
		} else {
			HR2GLE(UE_D3DDEVICE, hr);
		}

		hr = m.pD3D->CheckDeviceFormatConversion(D3DADAPTER_DEFAULT,
			D3DDEVTYPE_HAL, D3DFMT_YUY2, m.d3dFormat);
		if(hr == D3DERR_NOTAVAILABLE) {
			//SET_AND_SHOW_LE(UE_INSUFFICIENT_D3D_CAPS);
			TGLE(do_gdi_thing(m.hWnd));
		} else {
			HR2GLE(UE_D3DDEVICE, hr);
		}
	}

	if(g::software_yuyv) {
		HR2GLE(UE_INSUFFICIENT_D3D_CAPS, m.pD3D->CheckDeviceFormat(D3DADAPTER_DEFAULT,
			D3DDEVTYPE_HAL, m.d3dFormat, 0, D3DRTYPE_SURFACE, D3DFMT_X8R8G8B8));
		HR2GLE(UE_D3DDEVICE, m.pd3dDevice->CreateOffscreenPlainSurface(640, 480,
			D3DFMT_X8R8G8B8, D3DPOOL_DEFAULT, &m.pd3dSurface, NULL));
	} else {
		HR2GLE(UE_D3DDEVICE, m.pd3dDevice->CreateOffscreenPlainSurface(640, 480,
			D3DFMT_YUY2, D3DPOOL_DEFAULT, &m.pd3dSurface, NULL));
	}
	return true;
}

void CApp::cleanupD3D() {
	SAFE_RELEASE(m.pd3dSurface);
	SAFE_RELEASE(m.pd3dDevice);
	SAFE_RELEASE(m.pD3D);
}

bool CApp::initDI() {
	HR2GLE(UE_DIRECTINPUT, DirectInput8Create((HINSTANCE)GetWindowLong(m.hWnd,
		GWL_HINSTANCE), DIRECTINPUT_VERSION, IID_IDirectInput8, (void**)&m.pDI, NULL));
	HR2GLE(UE_DIRECTINPUT, m.pDI->CreateDevice(GUID_SysKeyboard, &m.pdiDevice, NULL)); 
	HR2GLE(UE_DIRECTINPUT, m.pdiDevice->SetDataFormat(&c_dfDIKeyboard));
	HR2GLE(UE_DIRECTINPUT, m.pdiDevice->SetCooperativeLevel(m.hWnd,
		DISCL_BACKGROUND | DISCL_NONEXCLUSIVE));
	HR2GLE(UE_DIRECTINPUT, SetBufferSize(m.pdiDevice, DIK_BUFFER_SIZE));

	HR2GLE(UE_DIRECTINPUT, m.pdiDevice->Acquire());

	return true;
}

void CApp::cleanupDI() {
	if(m.pdiDevice) m.pdiDevice->Unacquire();
	SAFE_RELEASE(m.pdiDevice);
	SAFE_RELEASE(m.pDI);
}

/*long CApp::_LM_RESET(HWND hWnd, WPARAM wParam, LPARAM lParam) {
static bool resetting = false;
if(!resetting) {
resetting = true;
R0GLE(quitThread());
if(!m.hw->gp_activated())
updateScreenAndKillTimer();
TheOneMessageBox("GameCube reset! Emulation stopped.");
resetting = false;
}
return 0;
}*/
void CApp::PostCloseMessage() {
	PostMessage(m.hWnd, WM_CLOSE, 0, 0);
	VDEGUB("WM_CLOSE posted\n");
}
long CApp::_WM_CLOSE(HWND hWnd, WPARAM, LPARAM) {
	VDEGUB("Handling WM_CLOSE...\n");
	GLE(quitThread());
	GLE(deleteCores());
	if(g::verbose) {
		//fails for no reason
		/*R0GLE*/(DestroyWindow(m.hwndVDO));
		/*R0GLE*/(DeleteObject(m.hfontCourierNew));
	}
	GLE(DestroyWindow(hWnd));
	return 0;
}
long CApp::_WM_DESTROY(HWND, WPARAM, LPARAM) {
	g_hWnd = NULL;
	PostQuitMessage(0);
	return 0;
}
long CApp::_WM_DROPFILES(HWND, WPARAM wParam, LPARAM) {
	DEGUB("File Dropped\n");
	HDROP hDrop = (HDROP)wParam;
	UINT nFiles = DragQueryFile(hDrop, MAX_DWORD, NULL, 0);
	if(nFiles != 1) {
		DEGUB("%i files dropped\n", nFiles);
		THE_ONE_MESSAGE_BOX("Bad bad multi-file dropper! ;}");
		return 0;
	}
	/*if(m.cpu) { //Do we need to do this?
	//MessageBox("You dropz0red a file! Wanna quit teh current one?");
	if(res == IDNO)
	return 0;
	}*/
	UINT fnSize = DragQueryFile(hDrop, 0, NULL, 0) + 1;
	Container<char> filename(fnSize);
	DragQueryFile(hDrop, 0, filename, fnSize);
	DragFinish(hDrop);
	DEGUB("%s\n", filename.p());

	GLE(quitThread());
	R0GLE(deleteCores());
	m.started_from_cmd_line = false;
	ProcessMessages();
	m_exename = filename;
	R0GLE(createCores());
	R0GLE(startThread());
	return 0;
}
long CApp::_WM_COMMAND(HWND hWnd, WPARAM wParam, LPARAM lParam) {
	if(HIWORD(wParam) > 1) {  //notification message
		WORD note = HIWORD(wParam);
		if(g::degub_msg) {
			DEGUB("Notification: %s | %08X\n", TranslateCN(note).c_str(), lParam);
		}
	} else switch(LOWORD(wParam)) {
	case IDM_ABOUT:
		DEGUB("Help->About\n");
		{
			ostringstream str;
			str << "WhineCube r" << RELEASE_NUMBER;
#ifdef BETA_VERSION
			str << " beta " << BETA_VERSION;
#endif
			str << "\nCopyright  Masken 2003-2005";
			R0GLE(MessageBox(hWnd, str.str().c_str(), "About", MB_ICONINFORMATION));
		}
		break;
	case IDM_OPEN:
		DEGUB("File->Open\n");
		m.started_from_cmd_line = false;
		R0GLE(quitThread());
		//#define FILE_TYPES(macro) macro(dol, DOL) macro(elf, ELF) macro(gcm, GCM)
		//#define 
		if(MyGetOpenFileName(hWnd, m_exename, "Open",
			"GameCube executables (*.dol; *.elf; *.gcm)\0*.dol;*.elf;*.gcm\0"
			"DOL files (*.dol)\0*.dol\0ELF files (*.elf)\0*.elf\0GCM files (*.gcm)\0*.gcm\0"))
		{
			R0GLE(deleteCores());
			ProcessMessages();
			R0GLE(createCores());
		}
		if(m.cpu) {
			R0GLE(startThread());
		}
		break;
	case IDM_CLOSE:
		DEGUB("File->Close\n");
		m.started_from_cmd_line = false;
		R0GLE(quitThread());
		R0GLE(deleteCores());
		break;
		/*case IDM_OPCODE:
		char buffer[80];
		char b2[80];
		DWORD cia, opcode;

		cia = 0x00000000;
		opcode = 0x4BFB22D9;
		//m.cpu->getdasmo(cia, opcode, buffer);
		sprintf(b2, "cia=0x%08X, opcode=0x%08X", cia, opcode); 
		TGLE(MessageBox(hWnd, buffer, b2, 0));
		break;*/
	case IDM_DEGUBBER:
		DEGUB("File->OpenDegubber\n");
		if(!m.hwndDegubber) {
			R0GLE(m.hwndDegubber = CreateDialog((HINSTANCE)GetWindowLong(hWnd,
				GWL_HINSTANCE), MAKEINTRESOURCE(IDD_DEGUB), hWnd, DegubberProc));
			ShowWindow(m.hwndDegubber, SW_SHOW);
		}
		break;
	case IDM_AOPTIONS:
		DEGUB("Options->Advanced Options...\n");
		{
			MYASSERT(!m.running);
			bool prev_softyuyv = g::software_yuyv;
			bool prev_rename = g::rename_degub;
			bool prev_degub_on = g::degub_on;
			/*if(m.running) {
			R0GLE(quitThread());
			}*/
			int res = DialogBox((HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE), 
				MAKEINTRESOURCE(IDD_AOPTIONS), hWnd, AOptionsProc);
			if(res == 0 || res == -1) {
				R0GLE(0);
			}
			if(prev_degub_on != g::degub_on) {
				if(g::degub_on) {
					R0GLE(g_degub.restart("degub.txt"));
					dumpStuffForRestart();
				} else {
					g_degub.turn_off();
					g_degub_write_mask = 0;
				}
			}
			if(prev_rename != g::rename_degub && !g::rename_degub) {
				R0GLE(g_degub.restart("degub.txt"));
				dumpStuffForRestart();
			}
			if(prev_softyuyv != g::software_yuyv) {
				SAFE_RELEASE(m.pd3dSurface);
				R0HR(createSurface());
			}
			R0GLE(setMenuChecks());
			/*if(was_running) {
			R0GLE(startThread());
			}*/
		}
		break;
	case IDM_PADS:
		DEGUB("Options->Configure Pads...\n");
		{
			//autopause should take care of this
			bool was_running = m.running;
			if(m.running) {
				R0GLE(quitThread());
			}
			int res = DialogBox((HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE), 
				MAKEINTRESOURCE(IDD_PADS), hWnd, PadsProc);
			if(res == 0 || res == -1) {
				R0GLE(0);
			}
			R0GLE(setMenuChecks());
			if(was_running) {
				R0GLE(startThread());
			}
		}
		break;
	case IDM_MEMCARDS:
		DEGUB("Options->Configure Memcards...\n");
		{
			bool was_running = m.running;
			if(m.running) {
				R0GLE(quitThread());
			}
			int res = DialogBox((HINSTANCE)GetWindowLong(hWnd, GWL_HINSTANCE), 
				MAKEINTRESOURCE(IDD_MEMCARDS), hWnd, MemcardsProc);
			if(res == 0 || res == -1) {
				R0GLE(0);
			}
			R0GLE(setMenuChecks());
			if(was_running) {
				R0GLE(startThread());
			}
		}
		break;
	case IDM_LOADGCM:
		DEGUB("GCM->LoadGCM\n");
		{
			string s;
			if(MyGetOpenFileName(hWnd, s, "Load GCM", FILTER_GCM)) {
				R0GLE(m.hw->loadgcm(s.c_str()));
				m.hw->setlid(false);
				R0GLE(setMenuChecks());
			}
		}
		break;
	case IDM_LIDOPEN:
		DEGUB("GCM->Lid open\n");
		m.hw->setlid(!m.hw->getlid());
		R0GLE(setMenuChecks());
		break;
	case IDM_SCREENSHOT:
		DEGUB("File->Snapshot\n");
		R0GLE(takeSnapshot(false));
		break;
	case IDM_PAUSE:
		DEGUB("Emulation->Paused\n");
		if(m.emu_error) {
			THE_ONE_MESSAGE_BOX("I don't wanna!");
		} else {
			if(m.running) {
				R0GLE(quitThread());
			} else {
				R0GLE(startThread());
			}
		}
		break;

#define SMB_COMMAND(mid, b) case mid: b = !b; R0GLE(setMenuChecks()); break;
		STANDARD_MENU_BOOLS(SMB_COMMAND);

	case IDM_WIREFRAME:
		{
			g::gp_wireframe = !g::gp_wireframe;
			R0GLE(setMenuChecks());
			static LPDIRECT3DBASETEXTURE9 old_textures[8] = {NULL};
			if(g::gp_wireframe) {
				for(int i=0; i<8; i++) {
					R0HR(m.pd3dDevice->GetTexture(i, old_textures + i));
					R0HR(m.pd3dDevice->SetTexture(i, NULL));
				}
			} else {
				for(int i=0; i<8; i++) {
					R0HR(m.pd3dDevice->SetTexture(i, old_textures[i]));
				}
			}
		}
		break;
	case IDM_HRR:
		DEGUB("Options->High refresh rate\n");
		g::hrr = !g::hrr;
		R0GLE(setMenuChecks());

		if(m.hw) {
			if(!m.hw->gp_activated()) {
				R0GLE(SetTimer(hWnd, SCREEN_TIMER, INIT_INTERVAL, NULL));
			}
		}
		break;
	case IDM_ALWAYSONTOP:
		DEGUB("Options->Always on top\n");
		g::always_on_top = !g::always_on_top;
		R0GLE(setMenuChecks());
		R0GLE(SetWindowPos(hWnd, g::always_on_top ? HWND_TOPMOST : HWND_NOTOPMOST,
			0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE));
		break;
	case IDM_EXIT:
		DEGUB("File->Exit\n");
		PostCloseMessage();
		break;
	case IDM_SAVESETTINGS:
		DEGUB("Options->Save\n");
		R0GLE(g_ini.WriteIniFile());
		break;
	case IDM_DUMPMEM:
		DEGUB("Emulation->Dumpmem\n");
		m.cpu->dumpmem();
		break;
	case IDM_VISI:
		DEGUB("Emulation->VISI\n");
		m.hw->doVISI();
		break;
	default:
		DEGUB("Unhandled WM_COMMAND!\n");
		R0GLE(MessageBox(hWnd,
			"This ain't implemented yet! You know who to bother about it. ;)",
			APP_NAME, MB_ICONEXCLAMATION));
	}
	return 0;
}
long CApp::_LM_DEGUBBERCLOSED(HWND, WPARAM, LPARAM) {
	m.hwndDegubber = NULL;
	return 0;
}

long CApp::_LM_DEBUG(HWND, WPARAM wParam, LPARAM lParam) {
	R0GLE(createDebugOutput());
	SendMessage(m.hwndDebugOutput, LM_DEBUG, wParam, lParam);
	return 0;
}
long CApp::_LM_DEBUG_OUTPUT_CLOSED(HWND, WPARAM, LPARAM) {
	m.hwndDebugOutput = NULL;
	return 0;
}

bool CApp::createDebugOutput() {
	if(m.hwndDebugOutput == NULL) {
		TGLE(m.hwndDebugOutput = CreateDialog((HINSTANCE)GetWindowLong(m.hWnd,
			GWL_HINSTANCE), MAKEINTRESOURCE(IDD_DEBUGOUTPUT), m.hWnd, DebugOutputProc));
		//put it on the right side of the main window
		RECT rect;
		TGLE(GetWindowRect(m.hWnd, &rect));
		TGLE(SetWindowPos(m.hwndDebugOutput, NULL,
			rect.left + 640 + GetSystemMetrics(SM_CXBORDER)*3, rect.top + 32,
			0, 0, SWP_NOACTIVATE | SWP_NOCOPYBITS | SWP_NOSIZE | SWP_NOZORDER));
		ShowWindow(m.hwndDebugOutput, SW_SHOW);
		//give focus back to the main window
		TGLE(SetActiveWindow(m.hWnd));
	}
	return true;
}

typedef BOOL (__stdcall * OFNfunc)(LPOPENFILENAME);

bool MyGetFileNameBase(HWND hwndOwner, string& s, const char *title,
											 const char *filter, const char *defext, DWORD extraFlags, OFNfunc func)
{
	OPENFILENAME ofn;
	ZeroMemory(&ofn, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hwndOwner;
	Container<char> buffer(MAX_PATH_BUFFER_SIZE(1));
	ofn.lpstrFile = buffer;
	buffer[0] = 0;
	ofn.nMaxFile = buffer.size();
	ofn.lpstrTitle = title;
	ofn.lpstrFilter = filter;
	ofn.lpstrDefExt = defext;
	ofn.Flags = extraFlags | OFN_HIDEREADONLY | OFN_NOCHANGEDIR |
		OFN_NONETWORKBUTTON;
	if(!func(&ofn)) {	//Cancel or error
		if(CommDlgExtendedError() != 0)	//user didn't press Cancel
			DoCDEE(CommDlgExtendedError());
		return false;
	} else {
		s = ofn.lpstrFile;
		return true;
	}
}

//Returns true if user selected a file, false otherwise.
bool MyGetOpenFileName(HWND hwndOwner, string& s, const char *title,
											 const char *filter)
{
	return MyGetFileNameBase(hwndOwner, s, title, filter, NULL,
		OFN_FILEMUSTEXIST, GetOpenFileName);
}

bool MyGetSaveFileName(HWND hwndOwner, string& s, const char *title,
											 const char *filter, const char *defext)
{
	return MyGetFileNameBase(hwndOwner, s, title, filter, defext,
		OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT, GetSaveFileName);
}

#define CHECK(boolean) ((boolean) ? MF_CHECKED : MF_UNCHECKED)
#define CMI(id, check) TGLECUSTOM(-1, CheckMenuItem(GetMenu(m.hWnd), id, MF_BYCOMMAND | (check)))
bool CApp::setMenuChecks() {
	if(IsWindow(m.hWnd)) {
		CMI(IDM_LIDOPEN, CHECK(m.hw ? m.hw->getlid() : false));
		CMI(IDM_PAUSE, CHECK(m.cpu && !m.running));
		CMI(IDM_HRR, CHECK(g::hrr));
		CMI(IDM_ALWAYSONTOP, CHECK(g::always_on_top));
		CMI(IDM_WIREFRAME, CHECK(g::gp_wireframe));
#define SMB_CMI(mid, b) CMI(mid, CHECK(b));
		STANDARD_MENU_BOOLS(SMB_CMI);
	}
	return true;
}

#define GRAY(r) ((r) ? MF_GRAYED : MF_ENABLED)
#define EMI(id, gray) TGLECUSTOM(-1, EnableMenuItem(GetMenu(m.hWnd), id, MF_BYCOMMAND | (gray)))
bool CApp::setMenuGreyness() {
	if(IsWindow(m.hWnd)) {
		EMI(IDM_AOPTIONS, GRAY(m.cpu));
		//EMI(IDM_SAVESETTINGS, GRAY(m.running));
		EMI(IDM_PAUSE, GRAY(!m.cpu || m.recompiling));
		//EMI(IDM_SAVESTATE, GRAY(!m.cpu));
		//EMI(IDM_LOADSTATE, GRAY(!m.cpu));
		EMI(IDM_LOADGCM, GRAY(!m.cpu));
		EMI(IDM_LIDOPEN, GRAY(!m.cpu));
		EMI(IDM_DUMPMEM, GRAY(!m.cpu));
		//EMI(IDM_VISIMPLE, GRAY(!m.cpu));
		EMI(IDM_SCREENSHOT, GRAY(!m.cpu));
		{
			bool b = false;
			if(m.hw) if(m.hw->gp_activated()) b = true;
			EMI(IDM_HRR, GRAY(b));
		}
		//EMI(IDM_VISI, GRAY(!m.cpu));
		EMI(IDM_CLOSE, GRAY(!m.cpu));
	}
	return true;
}

HWND init_window(HINSTANCE hInst) {
	WNDCLASS wc;
	HWND hWnd;

	wc.style = CS_DBLCLKS | CS_BYTEALIGNCLIENT;
	wc.lpfnWndProc = WndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hInstance = GetModuleHandle(NULL);
	wc.hIcon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_ICON2));
	wc.hCursor = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = NULL;//(HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wc.lpszMenuName = MAKEINTRESOURCE(IDR_MENU1);
	wc.lpszClassName = APP_NAME;
	if(!RegisterClass(&wc)) {
		RD((HWND)-4);
	}
	int cx = WIDTH + GetSystemMetrics(SM_CXFIXEDFRAME)*2;
	int cy = HEIGHT + GetSystemMetrics(SM_CYFIXEDFRAME)*2 +
		GetSystemMetrics(SM_CYMENU) + GetSystemMetrics(SM_CYSIZE);
	TGLE(hWnd = CreateWindowEx(WS_EX_ACCEPTFILES, APP_NAME, APP_NAME,
		WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | //WS_MAXIMIZEBOX |
		WS_VISIBLE, 0, 16, cx, cy, NULL, NULL, hInst, NULL));

	if(g::verbose) {
		RECT rect;
		GetClientRect(hWnd, &rect);
		DEGUB("Client size: %ix%i\n", rect.right+1, rect.bottom+1);
	}
	return hWnd;
}

// CreateStatusBar - creates a status bar and divides it into
// the specified number of parts. Also resizes the parent window.
// Returns the handle to the status bar.
// hwndParent - parent window for the status bar.
// nStatusID - child window identifier.
// hinst - handle to the application instance.
// nParts - number of parts into which to divide the status bar.
// parts - size in pixels of the individual parts.
HWND CreateStatusBar(HWND hwndParent, int nStatusID, HINSTANCE hinst,
										 int nParts, int *parts) {
											 HWND hwndStatus;

											 // Ensure that the common control DLL is loaded.
											 //InitCommonControls();

											 // Create the status bar.
											 TGLE(hwndStatus = CreateWindowEx(
												 0,                       // no extended styles
												 STATUSCLASSNAME,         // name of status bar class
												 (LPCTSTR)"",	     // no text when first created
												 WS_CHILD | WS_VISIBLE,   // creates a child window
												 0, 0, 0, 0,              // ignores size and position
												 hwndParent,              // handle to parent window
												 (HMENU) nStatusID,       // child window identifier
												 hinst,                   // handle to application instance
												 NULL));                  // no window creation data

											 // Tell the status bar to create the window parts.
											 SendMessage(hwndStatus, SB_SETPARTS, (WPARAM)nParts, (LPARAM)parts);

											 //Add some size to the parent window.
											 RECT mr, sr;
											 TGLE(GetWindowRect(hwndParent, &mr));
											 TGLE(GetWindowRect(hwndStatus, &sr));
											 TGLE(SetWindowPos(hwndParent, NULL, 0, 0, mr.right - mr.left,
												 (mr.bottom - mr.top) + (sr.bottom - sr.top),
												 SWP_NOACTIVATE | SWP_NOCOPYBITS | SWP_NOMOVE | SWP_NOZORDER));
											 TGLE(GetWindowRect(hwndParent, &mr));
											 if(g::verbose) {
												 DEGUB("WindowRect is now { %i, %i, %i, %i }. Size is %ix%i.\n", mr.left,
													 mr.top, mr.right, mr.bottom, mr.right - mr.left, mr.bottom - mr.top);
											 }

											 //Then we need to move the Status Bar
											 TGLE(SetWindowPos(hwndStatus, NULL, 0, HEIGHT, 0, 0,
												 SWP_NOACTIVATE | SWP_NOCOPYBITS | SWP_NOSIZE | SWP_NOZORDER));

											 return hwndStatus;
}  

//Returns false when WM_QUIT is recieved
bool ProcessMessages() {
	BOOL bRet;
	MSG msg;

	//Timing timing("ProcessMessages()");
	while(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE)) {
		if((bRet = GetMessage(&msg, NULL, 0, 0)) == 0)
			return false; //msg == WM_QUIT

		if(bRet == -1) {
			DEGUB("\nERROR in GetMessage\n");
			DO_GLE;
		} else {
			if(g::degub_msg) {
				DEGUB("Message recieved in ProcessMessages()\n");
			}
			if(!IsDialogMessage(g_capp.getDWnd(), &msg) &&
				!IsDialogMessage(g_capp.getDOWnd(), &msg))
			{
				if(!TranslateAccelerator(g_capp.getHWnd(), g_capp.getAccTable(), &msg)) {
					DispatchMessage(&msg);
				}
			}
		}
	}
	return true;
}

bool getNextNumberedName(string& s, const char *beginning, int digits,
												 const char *ending)
{
	MYASSERT(digits <= 9);
	WIN32_FIND_DATA wfd;
	HANDLE hFind;
	int max=0;
	char sscanfstring[MAX_PATH];	//unsafe!

	sprintf(wfd.cFileName, "%s%s%s", beginning, string(digits, '?').c_str(), ending);

	//sprintf(sscanfstring, "%s%%d%s", beginning, ending);
	{ //use strrchr to spruce up this section
		int j=0;
		for(size_t i=0; i<strlen(beginning); i++) {
			switch(beginning[i]) {
			case '\\':
				j = 0;
				break;
			case '%':
				sscanfstring[j++] = beginning[i];
				sscanfstring[j++] = beginning[i];
				break;
			default:
				sscanfstring[j++] = beginning[i];
			}
		}
		sscanfstring[j++] = '%';
		sscanfstring[j++] = 'd';
		for(size_t i=0; i<strlen(ending); i++) {
			switch(ending[i]) {
			case '\\':
				FAIL(UE_GENERIC_ERROR);
				break;
			case '%':
				sscanfstring[j++] = ending[i];
				sscanfstring[j++] = ending[i];
				break;
			default:
				sscanfstring[j++] = ending[i];
			}
		}
		sscanfstring[j] = 0;
	}

	hFind = FindFirstFile(wfd.cFileName, &wfd);
	if(hFind != INVALID_HANDLE_VALUE) {
		do {
			int i;
			if(sscanf_s(wfd.cFileName, sscanfstring, &i) == 1)
				if(i > max)
					max = i;
			if(!FindNextFile(hFind, &wfd)) {
				if(GetLastError() == ERROR_NO_MORE_FILES) {
					max++;
					break;
				} else
					FAIL(GetLastError());
			}
		} while(hFind != INVALID_HANDLE_VALUE);
		FindClose(hFind);
	}
	if(max == 0)
		max = 1;
	ostringstream str;
	str << beginning << setfill('0') << setw(digits) << max << ending;
	s = str.str();
	return true;
}

bool dumpToNumberedFile(const void *data, size_t size, const char *beginning,
												int digits, const char *ending)
{
	string filename;
	TGLE(getNextNumberedName(filename, beginning, digits, ending));
	TGLE(dumpToFile(data, size, filename.c_str()));
	return true;
}

bool dumpToFile(const void *data, size_t size, const char *filename) {
	DEGUB("Dumping %i bytes to %s\n", size, filename);
	FILE *file = NULL;
	if(fopen_s(&file, filename, "wb")) {
		FAIL(UE_CANNOT_OPEN_FILE);
	}
	MY_FWRITE(data, 1, size, file);
	fclose(file);
	return true;
}

void save_dib_to_png_file(const char *png_filename, LPBITMAPINFOHEADER dib,
													int dib_size, const void *pbits, int bitssize)
{
	PNGDIB *pngdib;

	pngdib = pngdib_d2p_init();
	pngdib_d2p_set_dib(pngdib,dib,dib_size,pbits,bitssize);
	pngdib_d2p_set_png_filename(pngdib,png_filename);
	pngdib_d2p_run(pngdib);
	pngdib_done(pngdib);
}

bool writePNG(const char *filename, WORD ByteCount, const void *data,
							int width, int height)
{
	BITMAPINFOHEADER bmh;
	ZERO_OBJECT(bmh);
	bmh.biSize = sizeof(BITMAPINFOHEADER);
	bmh.biWidth = width;
	bmh.biHeight = -height;
	bmh.biBitCount = ByteCount * 8;
	bmh.biCompression = BI_RGB;
	bmh.biPlanes = 1;
	bmh.biSizeImage = width*height*ByteCount;
	int totalsize = sizeof(BITMAPINFOHEADER) + bmh.biSizeImage;
	DEGUB("Writing PNG file %s\n", filename);
	save_dib_to_png_file(filename, &bmh, totalsize, data, bmh.biSizeImage);
	return true;
}

#include <png.h>
#include <SETJMP.H>
void __cdecl writepng_error_handler(png_structp png_ptr, png_const_charp msg) {
	DEGUB("writepng libpng error: %s\n", msg);
	_flushall();

	//png_structp error_ptr = (png_structp)png_get_error_ptr(png_ptr);
	if (png_ptr == NULL) {         /* we are completely hosed now */
		DEGUB("writepng severe error:  jmpbuf not recoverable; terminating.\n");
		_flushall();
		exit(99);
	}
	longjmp(png_ptr->jmpbuf, 1);
}
bool writePNGEx(const char *basename, int color_type, int width, int height,
								const void *data)
{
	int ByteCount;
#define SET_BC(type, count) case type: ByteCount = count; break;
#define TYPES(macro) macro(PNG_COLOR_TYPE_GRAY, 1) macro(PNG_COLOR_TYPE_GRAY_ALPHA, 2)\
	macro(PNG_COLOR_TYPE_PALETTE, 1) macro(PNG_COLOR_TYPE_RGB, 3)\
	macro(PNG_COLOR_TYPE_RGB_ALPHA, 4)
	switch(color_type) {
		TYPES(SET_BC);
	default:
		FAIL(UE_GENERIC_ERROR);
	}
	TGLE(writePNGEx2(basename, color_type, width, height, data, ByteCount*width,
		PNG_TRANSFORM_IDENTITY));
	return true;
}
bool writePNGEx2(const string& basename, int color_type, int width, int height,
								 const void *data, int pitch, int transforms)
{
	string filename = basename + ".png";
	DEGUB("Writing PNG file %s\n", filename.c_str());
	FILE *file = NULL;
	if(fopen_s(&file, filename.c_str(), "wb")) {
		FAIL(UE_CANNOT_OPEN_FILE);
	}
	png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		NULL, writepng_error_handler, NULL);
	if(!png_ptr) {
		fclose(file);
		FAIL(UE_GENERIC_ERROR);
	}
	png_infop info_ptr = png_create_info_struct(png_ptr);
	if(!info_ptr) {
		png_destroy_write_struct(&png_ptr, (png_infopp)NULL);
		fclose(file);
		FAIL(UE_GENERIC_ERROR);
	}
#pragma warning(push)
#pragma warning(disable:4611)
	if(setjmp(png_jmpbuf(png_ptr))) {
#pragma warning(pop)
		png_destroy_write_struct(&png_ptr, &info_ptr);
		fclose(file);
		FAIL(UE_GENERIC_ERROR);
	}
	png_init_io(png_ptr, file);
	png_set_IHDR(png_ptr, info_ptr, width, height, 8, color_type,
		PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);
	png_set_packing(png_ptr);
	png_set_compression_level(png_ptr, Z_BEST_COMPRESSION);

	Array<png_byte *> row_pointers(height);
	for(int i=0; i<height; i++) {
		row_pointers[i] = (png_byte *)data + i*pitch;
	}
	png_set_rows(png_ptr, info_ptr, row_pointers.p());
	png_write_png(png_ptr, info_ptr, transforms, NULL);

	png_destroy_write_struct(&png_ptr, &info_ptr);
	fclose(file);
	return true;
}
/*bool writeARGB2PNG(char *filename, int width, int height, const void *data, int pitch) {
}*/

/*bool dumpBitmap3(BITMAPINFOHEADER *bmh, const BYTE *bitmap) {
int memsize = sizeof(BITMAPINFOHEADER) + bmh->biSizeImage;
char filename[MAX_PATH];
TGLE(getNextNumberedName(filename, "wcs", 4, ".png"));
save_dib_to_png_file(filename, bmh, memsize, bitmap, bmh->biSizeImage);
return true;
}*/

bool snapshotCore(const BYTE *pRGB, WORD ByteCount) {
	string filename;
	TGLE(getNextNumberedName(filename, "wcs", 4, ".png"));
	TGLE(writePNG(filename.c_str(), ByteCount, pRGB, 640, 480));
	return true;
}

/*class ThreadSuspender {
public:
ThreadSuspender(HANDLE hThread) : h(hThread) { SuspendThread(h); }
~ThreadSuspender() { ResumeThread(h); }
private:
HANDLE h;
};
class ThreadPauser {
public:
ThreadPauser() : running(g_capp.m.running) { GLE(g_capp.quitThread()); }
~ThreadPauser() { if(running) { GLE(g_capp.startThread()); } }
private:
bool running;
};*/

bool CApp::takeSnapshot(bool direct) {
	Timing timing("Snapshot");
	D3DLOCKED_RECT locked_rect;

	bool was_running = m.running;
	TGLE(g_capp.quitThread());
	//*(int*)(0) = 0; //crash test
	TGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"Snapshot..."));

	if(!m.hw)
		FAIL(UE_GENERIC_ERROR);
	if(m.hw->gp_activated()) {  //3D
		LPDIRECT3DSURFACE9 pSurface;
		//if(direct) {
		//R1HR(m.pd3dDevice->GetRenderTarget(0, &pSurface));
		//R1HR(m.pd3dDevice->GetRenderTargetData());
		//} else {
		//}

		R1HR(m.pd3dDevice->CreateOffscreenPlainSurface(1024, 768, D3DFMT_A8R8G8B8,
			D3DPOOL_SCRATCH, &pSurface, NULL));
		R1HR(m.pd3dDevice->GetFrontBufferData(0, pSurface));

		RECT rect;
		POINT origin = { 0, 0 };
		TGLE(ClientToScreen(m.hWnd, &origin));
		rect.left = origin.x;
		rect.top = origin.y;
		rect.right = origin.x + WIDTH;
		rect.bottom = origin.y + HEIGHT;

		R1HR(pSurface->LockRect(&locked_rect, &rect, D3DLOCK_READONLY));
		Container<BYTE> con(WIDTH*HEIGHT*4);
		for(size_t y=0; y<HEIGHT; y++) {
			memcpy(con + y * WIDTH*4,
				((BYTE*)locked_rect.pBits) + y * locked_rect.Pitch, WIDTH*4);
		}
		TGLE(snapshotCore(con, 4));
		R1HR(pSurface->UnlockRect());
		pSurface->Release();
	} else { //2D
		if(direct) {
			R1HR(m.pd3dSurface->LockRect(&locked_rect, NULL, D3DLOCK_DISCARD));
			R1HR(updateFrameBuffer());
		} else {
			R1HR(m.pd3dSurface->LockRect(&locked_rect, NULL, D3DLOCK_READONLY));
		}

		if(g::software_yuyv) {
			TGLE(snapshotCore((BYTE *)locked_rect.pBits, 4));
		} else {
			Container<BYTE> screen_rgb(640*480*3);
			if(locked_rect.Pitch != 640*2) {
				DEGUB("Bad pitch: %i\n", locked_rect.Pitch);
				FAIL(UE_GENERIC_ERROR);
			}
			convertYUYV2RGB(screen_rgb, (BYTE *)locked_rect.pBits);
			TGLE(snapshotCore(screen_rgb, 3));
		}

		R1HR(m.pd3dSurface->UnlockRect());

		//m.snapshot_taken = true;
	}
	TGLE(SendMessage(m.hwndStatusBar, SB_SETTEXT, SB_MIPS, (LPARAM)"Paused"));
	if(was_running) {
		TGLE(g_capp.startThread());
	}
	return true;
}

void CApp::SuspendEmuThread() {
	SuspendThread(m.hThread);
}

void CApp::ResumeEmuThread() {
	ResumeThread(m.hThread);
}

void emuDebug(const char* buffer) {
	static FILE* file = NULL;
	if(g::separate_emudebug_file) {
		if(!file)
			fopen_s(&file, "emudebug.txt", "w");
		else
			fopen_s(&file, "emudebug.txt", "a");
		fprintf(file, "%s", buffer);
		fclose(file);
	}
}
